<?php
/**
 * Plugin Name: WP Google Analytics
 * Plugin URI: http://xavisys.com/wordpress-google-analytics-plugin/
 * Description: Lets you use <a href="http://analytics.google.com">Google Analytics</a> to track your WordPress site statistics
 * Version: 1.1.0
 * Author: Aaron D. Campbell
 * Author URI: http://xavisys.com/
 */

/**
 * Changelog:
 * 04/26/2008: 1.1.0
 *  - Major revamp to work better with the new Google Tracking Code.  It seems that outgoing links weren't being tracked properly.
 *
 * 04/17/2008: 1.0.0
 *  - Added to wordpress.org repository
 *
 * 07/03/2007: 0.2
 *  - Fixed problem with themes that do not call wp_footer().  If you are reading this and you are a theme developer, USE THE HOOKS!  That's what they're there for!
 *  - Updated how the admin section is handled
 *
 * 02/21/2007: 0.1
 *  - Original Version
 */

/*  Copyright 2006  Aaron D. Campbell  (email : wp_plugins@xavisys.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

//If this is set to true, extra info will be dumped to the browser.
//ONLY do this if you really need it
define('WGA_DEBUG', false);

/**
 * wpGoogleAnalytics is the class that handles ALL of the plugin functionality.
 * It helps us avoid name collisions
 * http://codex.wordpress.org/Writing_a_Plugin#Avoiding_Function_Name_Collisions
 */
class wpGoogleAnalytics {

	/**
	 * This adds the options page for this plugin to the Options page
	 */
	function admin_menu() {
		add_options_page(__('Google Analytics'), __('Google Analytics'), 'manage_options', str_replace("\\", "/", __FILE__), array('wpGoogleAnalytics', 'options'));
	}

	/**
	 * This is used to display the options page for this plugin
	 */
	function options() {
		/**
		 * @var WP_Roles
		 */
		global $wp_roles;

		//Deal with the data when submitted
		if (isset($_POST['wga_submit'])) {
			//Google Analytics Code
			$_POST['wga']['code'] = stripslashes($_POST['wga']['code']);
			//Special logging of 404s
			$_POST['wga']['log_404s'] = (!isset($_POST['wga']['log_404s']) || strtolower($_POST['wga']['log_404s']) != 'true')? 'false':'true';
			//Special logging of searches
			$_POST['wga']['log_searches'] = (!isset($_POST['wga']['log_searches']) || strtolower($_POST['wga']['log_searches']) != 'true')? 'false':'true';
			//Special logging of outgoing link
			$_POST['wga']['log_outgoing'] = (!isset($_POST['wga']['log_outgoing']) || strtolower($_POST['wga']['log_outgoing']) != 'true')? 'false':'true';
			//Ignore admin area
			$_POST['wga']['ignore_admin_area'] = (!isset($_POST['wga']['ignore_admin_area']) || strtolower($_POST['wga']['ignore_admin_area']) != 'true')? 'false':'true';
			//Add/remove wga_no_track capability for each role
			foreach ($wp_roles->roles as $role=>$role_info) {
				if (isset($_POST['wga']['role'][$role]) && $_POST['wga']['role'][$role] == 'true') {
					$wp_roles->add_cap($role, 'wga_no_track', true);
				} else {
					$wp_roles->add_cap($role, 'wga_no_track', false);
				}
			}
			//We don't need to store this in our setting
			unset($_POST['wga']['role']);
			//update our option
			update_option('wga', $_POST['wga']);
			echo '<div class="updated fade"><p>'.__('Options Updated.').'</p></div>';
		}

		//Get our options
		$wga = wpGoogleAnalytics::get_options();
		//Echo debug info if needed
		if (WGA_DEBUG) {
			echo '<pre>',var_dump($wga),'</pre>';
		}
		//We will fill $roles with checkboxes to ignore each role
		$roles = '';
		foreach ($wp_roles->roles as $role=>$role_info) {
			$checked = (isset($role_info['capabilities']['wga_no_track']) && $role_info['capabilities']['wga_no_track'])? ' checked="checked"':'';
			$role_info['name'] .= (strtolower(substr($role_info['name'], -1)) != 's')? 's':'';
			$roles .= "					<label for='wga_role_{$role}'><input type='checkbox' name='wga[role][{$role}]' value='true' id='wga_role_{$role}'{$checked} /> ".__("Do not log {$role_info['name']} when logged in")."</label><br />";
		}
?>
		<div class="wrap">
			<h2><?php _e('Google Analytics Options') ?></h2>
			<form action="" method="post" id="wp_google_analytics">
				<fieldset class="options">
					<legend><?php _e('Google Analytics Code'); ?></legend>
					<label for="wga_code">
						<?php _e('Paste your <a href="http://analytics.google.com">Google Analytics</a> code into the textarea:'); ?><br /><br />
						<textarea name="wga[code]" id="wga_code" style="width:400px; height:150px;"><?php echo htmlentities($wga['code']); ?></textarea><br /><br />
					</label>
					<label for="wga_ignore_admin_area"><input type="checkbox" name="wga[ignore_admin_area]" value="true" id="wga_ignore_admin_area" <?php if ($wga['ignore_admin_area'] == 'true') { echo ' checked="checked"';} ?> /> <?php _e('Do not log anything in the admin area'); ?></label><br />
					<label for="wga_log_404s"><input type="checkbox" name="wga[log_404s]" value="true" id="wga_log_404s" <?php if ($wga['log_404s'] == 'true') { echo ' checked="checked"';} ?> /> <?php _e('Log 404 errors as /404/{url}?referrer={referrer}'); ?></label><br />
					<label for="wga_log_searches"><input type="checkbox" name="wga[log_searches]" value="true" id="wga_log_searches" <?php if ($wga['log_searches'] == 'true') { echo ' checked="checked"';} ?> /> <?php _e('Log searches as /search/{search}?referrer={referrer}'); ?></label><br />
					<label for="wga_log_outgoing"><input type="checkbox" name="wga[log_outgoing]" value="true" id="wga_log_outgoing" <?php if ($wga['log_outgoing'] == 'true') { echo ' checked="checked"';} ?> /> <?php _e('Log outgoing links as /outgoing/{url}?referrer={referrer}'); ?></label><br />
<?php echo $roles; ?>
				</fieldset>
				<p class="submit">
					<input type="submit" name="wga_submit" value="Update Options &raquo;" />
				</p>
			</form>
		</div>
<?php
	}

	/**
	 * Used to generate a tracking URL
	 *
	 * @param array $track - Must have ['data'] and ['code']
	 * @return string - Tracking URL
	 */
	function get_url($track) {
		$site_url = (($_SERVER['HTTPS'] == 'on')? 'https://':'http://').$_SERVER['HTTP_HOST'];
		foreach ($track as $k=>$value) {
			if (strpos(strtolower($value), strtolower($site_url)) === 0) {
				$track[$k] = substr($track[$k], strlen($site_url));
			}
			if ($k == 'data') {
				$track[$k] = preg_replace("/^https?:\/\/|^\/+/i", "", $track[$k]);
			}

			//This way we don't lose search data.
			if ($k == 'data' && $track['code'] == 'search') {
				$track[$k] = urlencode($track[$k]);
			} else {
				$track[$k] = preg_replace("/[^a-z0-9\.\/\+\?=-]+/i", "_", $track[$k]);
			}

			$track[$k] = trim($track[$k], '_');
		}
		$char = (strpos($track['data'], '?') === false)? '?':'&';
		return str_replace("'", "\'", "/{$track['code']}/{$track['data']}{$char}referer={$_SERVER['HTTP_REFERER']}");
	}

	/**
	 * This injects the Google Analytics code into the footer of the page.
	 *
	 * @param bool[optional] $output - defaults to true, false returns but does NOT echo the code
	 */
	function insert_code($output=true) {
		//If $output is not a boolean false, set it to true (default)
		$output = ($output !== false);
		//This sets a static variable that says if the code has been inserted.
		wpGoogleAnalytics::code_inserted_static(true);
		//get our plugin options
		$wga = wpGoogleAnalytics::get_options();
		//If the user's role has wga_no_track set to true, return without inserting code
		if (current_user_can('wga_no_track')) {
			$ret = "<!-- Google Analytics Plugin is set to ignore your user role -->\r\n";
			if ($output) {
				echo $ret;
			}
			return $ret;
		}
		//If the Google Analytics code has been set
		if ($wga['code'] !== false) {
			//Echo debug info if needed
			if (WGA_DEBUG) {
				echo '<pre>',var_dump($wga),'</pre>';
			}

			//If $admin is true (we're in the admin_area), and we've been told to ignore_admin_area, return without inserting code
			if (is_admin() && (!isset($wga['ignore_admin_area']) || $wga['ignore_admin_area'] != 'false')) {
				$ret = "<!-- Your Google Analytics Plugin is set to ignore Admin area -->\r\n";
				if ($output) {
					echo $ret;
				}
				return $ret;
			} elseif (is_404() && (!isset($wga['log_404s']) || $wga['log_404s'] != 'false')) {
				//Set track for 404s, if it's a 404, and we are supposed to
				$track['data'] = $_SERVER['REQUEST_URI'];
				$track['code'] = '404';
			} elseif (is_search() && (!isset($wga['log_searches']) || $wga['log_searches'] != 'false')) {
				//Set track for searches, if it's a search, and we are supposed to
				$track['data'] = $_REQUEST['s'];
				$track['code'] = "search";
			}

			//If we need to generate a special tracking URL
			if (isset($track)) {
				//get the tracking URL
				$track['url'] = wpGoogleAnalytics::get_url($track);

				//adjust the code that we output, account for both types of tracking
				$wga['code'] = str_replace("urchinTracker()","urchinTracker('{$track['url']}')", $wga['code']);
				$wga['code'] = str_replace("pageTracker._trackPageview()","pageTracker._trackPageview('{$track['url']}')", $wga['code']);

				//Echo debug info if needed
				if (WGA_DEBUG) {
					echo '<pre>',var_dump($track, $site_url),'</pre>';
				}
			}
			//output the Google Analytics code
			if ($output) {
				echo $wga['code'];
			}
			return $wga['code'];
		} else {
			//If the Google Analytics code has not been set in admin area, return without inserting code
			$ret = "<!-- You need to set up the Google Analytics Plugin -->\r\n";
			if ($output) {
				echo $ret;
			}
			return $ret;
		}
	}

	/**
	 * Used to get one or all of our plugin options
	 *
	 * @param string[optional] $option - Name of options you want.  Do not use if you want ALL options
	 * @return array of options, or option value
	 */
	function get_options($option = null) {
		$o = get_option('wga');
		if (isset($option)) {
			if (isset($o[$option])) {
				return $o[$option];
			} else {
				return false;
			}
		} else {
			return $o;
		}
	}

	/**
	 * Start our output buffering with a callback, to grab all links
	 *
	 * @todo If there is a good way to tell if this is a feed, add a seperate option for tracking outgoings on feeds
	 */
	function start_ob() {
		$log_outgoing = wpGoogleAnalytics::get_options('log_outgoing');
		//Only start the output buffering if we care, and if it's NOT an XMLRPC REQUEST
		if (($log_outgoing == 'true' || $log_outgoing === false) && (!defined('XMLRPC_REQUEST') || !XMLRPC_REQUEST)) {
			ob_start(array('wpGoogleAnalytics', 'get_links'));
		}
	}

	/**
	 * Grab all links on the page.  If the code hasn't been inserted, we want to
	 * insert it just before the </body> tag
	 *
	 * @param string $b - buffer contents
	 * @return string - modified buffer contents
	 */
	function get_links($b) {
		$b = preg_replace_callback("/
			<\s*a							# anchor tag
				(?:\s[^>]*)?		# other attibutes that we don't need
				\s*href\s*=\s*	# href (required)
				(?:
					\"([^\"]*)\"	# double quoted link
				|
					'([^']*)'			# single quoted link
				|
					([^'\"\s]*)		# unquoted link
				)
				(?:\s[^>]*)?		# other attibutes that we don't need
				\s*>						#end of anchor tag
			/isUx", array('wpGoogleAnalytics', 'handle_link'), $b);
		//If the code hasn't been inserted, take drastic measures and add it here
		if (!wpGoogleAnalytics::code_inserted_static()) {
			$b = str_replace('</body>', wpGoogleAnalytics::insert_code(false).'</body>', $b);
		}
		return $b;
	}

	/**
	 * If a link is outgoing, add an onclick that runs some Google JS with a
	 * generated URL
	 *
	 * @param array $m - A match from the preg_replace_callback in self::get_links
	 * @return string - modified andchor tag
	 */
	function handle_link($m) {
		$code = wpGoogleAnalytics::get_options('code');
		//get our site url...used to see if the link is outgoing.  We can't use the wordpress setting, because wordpress might not be running at the document root.
		$site_url = (($_SERVER['HTTPS'] == 'on')? 'https://':'http://').$_SERVER['HTTP_HOST'];
		$link = array_pop($m);
		//If the link is outgoing, we modify $m[0] (the anchor tag)
		if (preg_match("/^https?:\/\//i", $link) && (strpos(strtolower($link), strtolower($site_url)) !== 0 )) {
			//get our custom link
			$track['data'] = $link;
			$track['code'] = 'outgoing';
			$track['url'] = wpGoogleAnalytics::get_url($track);

			// Check which version of the code the user is using, and user proper function
			$function = (strpos($code, 'ga.js') !== false)? 'pageTracker._trackPageview': 'urchinTracker';
			$onclick = "{$function}('{$track['url']}');";

			//If there is already an onclick, add to the beginning of it (adding to the end will not work, because too many people leave off the ; from the last statement)
			if (preg_match("/onclick\s*=\s*(['\"])/iUx",$m[0],$match)) {
				//If the onclick uses single quotes, we use double...and vice versa
				if ($match[1] == "'" ) {
					$onclick = str_replace("'", '"', $onclick);
				}
				$m[0] = str_replace($match[0], $match[0].$onclick, $m[0]);
			} else {
				$m[0] = str_replace('>', " onclick=\"{$onclick}\">", $m[0]);
			}
		}
		//return the anchor tag (modified or not)
		return $m[0];
	}

	/**
	 * This is a lame PHP4 hack since PHP4 doesn't support static class variables.
	 * One of these days I'm just going to say "screw people that won't upgrade"
	 * and do ONLY PHP5
	 *
	 * @param mixed $set[optional] - What to set the static var to
	 * @return mixed - value of static var
	 */
	function code_inserted_static($set = null) {
		static $static = false;
		if (!is_null($set)) {
			$static = $set;
		}
		return $static;
	}
}

/**
 * Add the necessary hooks
 */
add_action('admin_menu', array('wpGoogleAnalytics','admin_menu'));
add_action('wp_footer', array('wpGoogleAnalytics', 'insert_code'));
add_action('init', array('wpGoogleAnalytics', 'start_ob'));
?>