<?php
/*
Plugin Name: Referrer Detector
Plugin URI: http://www.phoenixheart.net/2008/11/referrer-detector
Description: Helps your blog detect where the user comes from and automatically displays the corresponding greeting.
Version: 3.2
Author: Phoenixheart
Author URI: http://www.phoenixheart.net
*/

global $rdetector_table_name, $rdetector_stats_table_name, $rdetector_db_version, $rdetector_text_domain, $rdetector_plugin_dir,
	$rdetector_wpadmin_index_page, $rdetector_help_url, $rdetector_options, $wpdb;

$rdetector_plugin_name = 'Referrer Detector';
$rdetector_table_name = "{$wpdb->prefix}referrer_detector";
$rdetector_stats_table_name = "{$wpdb->prefix}referrer_detector_stats";
$rdetector_db_version = '2.0';
$rdetector_text_domain = 'referrer_detector';
$rdetector_plugin_dir = get_option('siteurl') . '/wp-content/plugins/referrer-detector/';
$rdetector_wpadmin_index_page = get_option('siteurl') . '/wp-admin/index.php';
$rdetector_help_url = 'http://www.phoenixheart.net/2008/11/referrer_detector';
define('RDETECTOR_CUSTOM_TAG', '{referrer_detector}');
define('RDETECTOR_NONCE_ACTION', md5($rdetector_plugin_name . $rdetector_wpadmin_index_page));

load_plugin_textdomain($rdetector_text_domain);

$rdetector_add_help = '<p>To add or edit an entry, you&#8217;re provided with a simple form with 5 fields.</p>
<ul>
	<li><strong>Name</strong>: The name of the entry. Digg, Google, Yahoo&#8230; are good names.</li>
	<li>
		<p><strong>URL</strong>: The URL of the site where your users come from. For this to work, the URL must not contain those prefix and suffix like &#8220;http:/www&#8221; and &#8220;/?q=etc.&#8221;. To be clear, it should be &#8220;digg.com&#8221; instead of &#8220;http://digg.com&#8221;, and &#8220;google.com&#8221; instead of &#8220;http://www.google.com/search?hl=en&amp;q=some+keywords&#8221;. Furthermore, the domain and subdomain are treated differently, which means different greeting boxes (if any) will be shown for &#8220;yahoo.com&#8221; and &#8220;search.yahoo.com&#8221; users.</p>
		<p>[IMPORTANT] Version 3.2 introduces two new features:</p>
		<ul>
			<li>The asterisk (*) character is supported as a wildcard. For example, google.* matches ALL Google localized domains too.</li>
			<li>Several related URLs can be combined into one, seperated by commas. For example a URL "del.icio.us,delicious.com" matches both domains.</li>
		</ul>
	</li>
	<li>
		<p><strong>Message</strong>: The greeting text for this entry. These HTML tags are allowed: <code>&lt;div&gt;&lt;h1&gt;&lt;h2&gt;&lt;h3&gt;&lt;p&gt;&lt;span&gt;&lt;a&gt;&lt;ul&gt;&lt;ol&gt;&lt;li&gt;&lt;hr&gt;&lt;br&gt;&lt;table&gt;&lt;thead&gt;&lt;tbody&gt;&lt;tfoot&gt;&lt;tr&gt;&lt;td&gt;&lt;th&gt;&lt;strong&gt;&lt;em&gt;</code> Also, these tags may be used:</p>
		<ul>
			<li><strong>{url} </strong>- Full URL to your WordPress post, for example http://yousite.com/2008/10/my-first-post. Very handy if you want to ask your users to digg your post.</li>
			<li><strong>{title} </strong>- Title of the post, for example My First Post.</li>
			<li><strong>{link}</strong> - Link to the post, for example <a href="http://yousite.com/2008/10/my-first-post">My First Post</a>. It&#8217;s just an combination of {url} and {title}.</li>
			<li><strong>{categorynames}</strong> - Names of the categories under which the post is listed, separated by a comma. Example: Cats, Dogs, Pets.</li>
			<li><strong>{categorylinks}</strong> - Links to the categories under which the post is listed, separated by a comma. Example: <a href="http://yousite.com/categories/cats">Cats</a>, <a href="http://yousite.com/categories/dogs">Dogs</a>, <a href="http://yousite.com/categories/pets">Pets</a></li>
			<li><strong>{authorname}</strong> - Name of the author of the post, for example Mr. Cool.</li>
			<li><strong>{authorurl}</strong> - URL of the author, for example http://mrcool.com.</li>
			<li><strong>{authorlink}</strong> - Link to the author, for example <a href="http://mrcool.com">Mr. Cool</a></li>
			<li><strong>{search-terms}</strong> - The search terms that user uses to find your blog, seperated by commas</strong></li>
		</ul>
	</li>
	<li><strong>Icon</strong>: The URL of the icon to display beside the greeting text. If this is left blank, a default RSS icon will be shown. Though any image will do, it&#8217;s recommended to use a small one (48&#215;48 px should be perfect).</li>
	<li><strong>This entry is active</strong>: The name says it all. Naturally, this should be checked.</li>
</ul>';

$rdetector_generic_help = '<p>Those options are used to specify if you want to show the greeting box on every post,
page, before or after&#8230; Notice that if you can include a custom tag <strong>{referrer_detector}</strong> anywhere
in your post to show the welcome text. If the visitor didn&#8217;t come from any of the referrers,
don&#8217;t worry, the plugin is smart enough to hide that ugly tag from your post.</p>
<p>Starting from version 2.0, a template tag <em>&lt;?php if (function_exists(\'referrer_detector\')) referrer_detector(); ?&gt;</em>
can be placed somewhere in the theme files (most likely index.php) to display a greet box too.</p>';

$rdetector_exclude_help = '<p>Here you can specify a list of urls (comma-seperated). If the visitor is comming from one of these urls, the welcome box will not be show. <br />
For example, by adding <em>google.com/reader</em> into the list, you make sure that the readers won\'t see the box,
which is good since they should have registered to your feed already.</p>
<p>Starting from version 3.2, as of referrer URL, you can use the wildcard character (*) on this field too: <em>google.*/reader</em> matches the readers of all localized Google domains.</p>';


$rdetector_options = array(
	'rdetector_add_to_every_post' => array(
		'type'			=> 'select',
		'label' 		=> __('Automatically display welcome message on every post? *', $rdetector_text_domain),
		'options'		=> array ('yes', 'no'),
		'default_value'	=> 'yes',
	),
	'rdetector_add_to_every_page' => array(
		'type'			=> 'select',
		'label' 		=> __('Automatically display welcome message on every page? *', $rdetector_text_domain),
		'options'		=> array ('yes', 'no'),
		'default_value'	=> 'no',
	),
	'rdetector_message_position' => array(
		'type'			=> 'select',
		'label' 		=> __('The position of the welcome message (related to page/post content)', $rdetector_text_domain),
		'options'		=> array ('before', 'after', 'both'),
		'default_value'	=> 'before',
	),
	'rdetector_close_icon' 	=> array(
		'type'			=> 'select',
		'label' 		=> __('Display an icon to let user close the message?', $rdetector_text_domain),
		'options'		=> array ('yes', 'no'),
		'default_value'	=> 'yes',
	),
);

/**
 * @desc	Creates a new table to hold the data
 */
function rdetector_install()
{
	global $wpdb, $rdetector_table_name, $rdetector_stats_table_name, $rdetector_db_version, $rdetector_plugin_dir;

	require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

	if ($wpdb->get_var("SHOW TABLES LIKE '$rdetector_table_name'") != $rdetector_table_name)
	{

		$sql = "CREATE TABLE $rdetector_table_name (
		  `id` smallint(5) UNSIGNED NOT NULL AUTO_INCREMENT,
		  `name` varchar(50) NOT NULL,
		  `url` varchar(255) NOT NULL,
		  `icon` varchar(255) NOT NULL,
		  `welcome_text` text NOT NULL,
		  `enabled` tinyint(4) NOT NULL default '1',
		  PRIMARY KEY (`id`),
		  KEY `url` (`url`)
		);";

		dbDelta($sql);
		rdetector_add_default_entries();
	}

	if ($wpdb->get_var("SHOW TABLES LIKE '$rdetector_stats_table_name'") != $rdetector_stats_table_name)
	{
		// create the stats table
    	$sql = "CREATE TABLE `wp_referrer_detector_stats` (
          `id` bigint(20) NOT NULL auto_increment,
          `referrer` varchar(50) collate latin1_general_ci NOT NULL,
          `time` datetime NOT NULL,
          PRIMARY KEY  (`id`),
          KEY `referrer` (`referrer`)
        );";

		dbDelta($sql);
	}

	rdetector_generate_js_data();

	add_option('referrer_detector_db_version', $rdetector_db_version);
}

# Hook into the activation process to create the tables
register_activation_hook(__FILE__, 'rdetector_install');

/**
* @desc 	Adds some entries as default
*/
function rdetector_add_default_entries()
{
	global $wpdb, $rdetector_table_name, $rdetector_plugin_dir;

	$rss_link = get_bloginfo('rss2_url');

	$sql = "INSERT INTO `$rdetector_table_name` (`name`, `url`, `icon`, `welcome_text`, `enabled`) VALUES
	('Google', 'google.com', '{$rdetector_plugin_dir}images/icons/google.png', 'Welcome <strong>Googler</strong>! If you find this page useful, why not <a href=\"$rss_link\">subscribe to the RSS feed</a> for more interesting posts in the future?', 1),
	('Digg', 'digg.com', '{$rdetector_plugin_dir}images/icons/digg.png', 'Hello fellow <strong>Digger</strong>! If you find this story useful, please <a href=\"http://digg.com/submit?url={url}\">digg it</a>!', 1),
	('StumbleUpon', 'stumbleupon.com', '{$rdetector_plugin_dir}images/icons/stumbleupon.png', 'Hello fellow <strong>Stumbler</strong>! Don\'t forget to <a href=\"http://www.stumbleupon.com/submit?url={url}\" rel=\"nofollow\">give me a thumbs up</a> if you like this page!', 1),
	('Technograti', 'technorati.com', '{$rdetector_plugin_dir}images/icons/technorati.png', 'Hello fellow <strong>Technorati</strong> user! Don\'t forget to <a href=\"http://technorati.com/faves?add={url}\" rel=\"nofollow\">favorite this blog</a> if you like it!', 1),
	('Twitter', 'twitter.com', '{$rdetector_plugin_dir}images/icons/twitter.png', 'Hello fellow <strong>Twitter</strong> user! Don\'t forget to <a href=\"http://twitthis.com/twit?url={url}\" rel=\"nofollow\">Twit this post</a> if you like it, or <a href=\"http://twitter.com/\" rel=\"nofollow\">follow me</a> on Twitter if you find me interesting.', 1),
	('Del.icio.us', 'del.icio.us', '{$rdetector_plugin_dir}images/icons/delicious.png', 'Hello fellow <strong>Delicious</strong> user! Feel free to <a href=\"http://delicious.com/post?url={url}\" rel=\"nofollow\">bookmark this page</a> for future reference if you like it!', 1),
	('Delicious.com', 'delicious.com', '{$rdetector_plugin_dir}images/icons/delicious.png', 'Hello fellow <strong>Delicious</strong> user! Feel free to <a href=\"http://delicious.com/post?url={url}\" rel=\"nofollow\">bookmark this page</a> for future reference if you like it!', 1),
	('Yahoo', 'search.yahoo.com', '{$rdetector_plugin_dir}images/icons/yahoo.png', 'Welcome fellow <strong>Yahooligan</strong>! If you find this page useful, you might want to <a href=\"$rss_link\" rel=\"nofollow\">subscribe to the RSS feed</a> for updates on this topic.', 1),
	('Live', 'search.live.com', '{$rdetector_plugin_dir}images/icons/live.png', 'Welcome fellow <strong>Live Search</strong> user! If you find this page useful, you might want to <a href=\"$rss_link\" rel=\"nofollow\">subscribe to the RSS feed</a> for updates on this topic.', 1),
	('Lifehacker', 'lifehacker.com', '', 'Hello fellow <strong>Lifehacker</strong> reader! If you find this page useful, you might want to <a href=\"$rss_link\" rel=\"nofollow\">subscribe to the RSS feed</a> for updates on this topic.', 1),
	('TechCrunch', 'techcrunch.com', '{$rdetector_plugin_dir}images/icons/techcrunch.png', 'Hello fellow <strong>TechCrunch</strong> reader! If you find this page useful, you might want to <a href=\"$rss_link\" rel=\"nofollow\">subscribe to the RSS feed</a> for updates on this topic.', 1),
	('9rules', '9rules.com', '{$rdetector_plugin_dir}images/icons/9rules.png', 'Hello fellow <strong>9rules</strong> reader! If you find this page useful, you might want to <a href=\"$rss_link\" rel=\"nofollow\">subscribe to the RSS feed</a> for updates on this topic.', 1),
	('Gizmodo', 'gizmodo.com', '', 'Hello fellow <strong>Gizmodo</strong> reader! If you find this page useful, you might want to <a href=\"$rss_link\" rel=\"nofollow\">subscribe to the RSS feed</a> for updates on this topic.', 1),
	('TechRadar', 'techradar.com', '', 'Hello fellow <strong>TechRadar</strong> reader! If you find this page useful, you might want to <a href=\"$rss_link\" rel=\"nofollow\">subscribe to the RSS feed</a> for updates on this topic.', 1),
	('Reddit', 'reddit.com', '{$rdetector_plugin_dir}images/icons/reddit.png', 'Hello fellow <strong>Reddit</strong> user! If you find this story useful, please <a href=\"http://www.reddit.com/submit?url={url}\">vote it up</a>!', 1);";

	$results = $wpdb->query($sql);
}

/**
 * @desc	Hooks into the admin page's header
 */
function rdetector_include_scripts()
{
	global $rdetector_plugin_dir;

	echo(
		'<link rel="stylesheet" href="' . $rdetector_plugin_dir . 'css/admin_style.css" type="text/css" media="screen" />' .
		'<script type="text/javascript" src="http://jqueryjs.googlecode.com/files/jquery-1.2.6.min.js"></script>' .
		'<script type="text/javascript" src="' . $rdetector_plugin_dir . 'js/jquery-ui-personalized.js"></script>' .
		'<script type="text/javascript" src="' . $rdetector_plugin_dir . 'js/jquery.livequery.js"></script>' .
		'<script type="text/javascript" src="' . $rdetector_plugin_dir . 'js/admin_functions.js"></script>' .
		'<script type="text/javascript" src="' . $rdetector_plugin_dir . 'js/admin_onload.js"></script>'
	);
}

/**
 * @desc	Creates the options form
 */
function rdetector_options_form()
{
	rdetector_include_scripts();

	global $rdetector_plugin_name, $rdetector_table_name, $wpdb,
	$rdetector_plugin_dir, $rdetector_text_domain, $rdetector_wpadmin_index_page,
	$rdetector_help_url, $rdetector_add_help, $rdetector_generic_help, $rdetector_exclude_help;

	printf('<div class="wrap">
		<input type="hidden" id="rdetectorNonce" name="nonce" value="%s" />
		<div id="divGenericResult" class="updated fade" style="display:none"></div>
		<h2>%s</h2>
		<ul>
			<li><a href="#divListContainer">%s</a></li>
			<li><a href="#divAddContainer">%s</a></li>
			<li><a href="#divEditContainer">%s</a></li>
			<li><a href="#divExcluded">%s</a></li>
			<li><a href="#divGenericOptions">%s</a></li>
			<li><a href="#divStats">%s</a></li>
		</ul>',
		wp_create_nonce(RDETECTOR_NONCE_ACTION),
		__('Referrer Detector Options', $rdetector_text_domain),
		__('Entry List', $rdetector_text_domain),
		__('Add Entry', $rdetector_text_domain),
		__('Edit Entry', $rdetector_text_domain),
		__('Excluded URLs', $rdetector_text_domain),
		__('Generic Options', $rdetector_text_domain),
		__('Stats', $rdetector_text_domain));

	// the current entries
	printf('
		<div id="divListContainer">
		<div class="tablenav">
			<div class="alignleft actions">
				<select name="action" id="bulkSelect" autocomplete="off">
					<option value="nope">Bulk Action</option>
					<option value="bulk-activate">Activate</option>
					<option value="bulk-deactivate">Deactivate</option>
					<option value="bulk-merge">Merge</option>
					<option value="bulk-delete">Delete</option>
				</select>
				<input class="button-secondary action" id="bulkSubmit" type="submit" value="Apply" autocomplete="off" />
			</div>
		</div>
		<div class="clear"></div>
			%s
		<p class="rdetector_restore"><a class="ajax need_confirm" confirmtext="%s" href="#" action="restore" entry="nope"><input type="button" class="button-secondary action" value="%s" /></a></p>
		<div id="loading_nope" style="display: none"><img src="%s/images/loading.gif" alt="Loading..." /></div>
	</div>',
		rdetector_get_entry_table(),
		__('Warning: This will delete ALL of your custom entries (if any)! Are you sure?'), __('Restore default entries'),
		$rdetector_plugin_dir);

	// the edit div
	printf('
	<div id="divEditContainer" style="display:none">
		<p id="editInstruction">%s</p>
		<p class="toggler"><span style="font-size:11px; font-weight:normal">%s</span></p>
		<div class="toggled help">%s</div>
		<form action="index.php" method="post" id="formReferrerDetector_Edit" autocomplete="off">
		</form>
		<div style="display:none" id="divLoadingEdit">
			<img src="%s/images/loading.gif" alt="Loading..." />
		</div>
	</div>',
	__('Click "View/Edit" on the entry list to edit.', $rdetector_text_domain),
	__('Get help with this', $rdetector_text_domain),
	__($rdetector_add_help, $rdetector_text_domain),
	$rdetector_plugin_dir);

	// the Add div
	printf('
	<div id="divAddContainer">
		<p class="toggler"><span style="font-size:11px; font-weight:normal">%s</span></p>
		<div class="toggled help">%s</div>
		<form action="index.php" method="post" id="formReferrerDetector_Add" autocomplete="off">
			<p>
				<label for="nameAdd">%s</label><br />
				<input type="text" name="name" id="nameAdd" class="required" />
			</p>
			<p>
				<label for="urlAdd">%s</label><br />
				<input type="text" name="url" id="urlAdd" class="required" />
			</p>
			<p>
				<label for="messageAdd">%s</label><br />
				<textarea name="message" id="messageAdd" class="required"></textarea>
			</p>
			<p>
				<label for="iconAdd">%s</label><br />
				<input type="text" name="icon" id="iconAdd" /> (%s)
			</p>
			<p>
				<input type="checkbox" name="enabled" id="enabledAdd" checked="checked" />
				<label for="enabledAdd">%s</label>
				<input type="hidden" name="rdetector_action" value="add" />
			</p>
			<div class="submit">
				<input type="submit" value="%s" />
			</div>
		</form>
		<div style="display: none" id="divLoadingAdd">
			<img src="%s/images/loading.gif" alt="Loading..." />
		</div>
	</div>
	',
	__('Get help with this', $rdetector_text_domain),
	__($rdetector_add_help, $rdetector_text_domain),
	__('Name', $rdetector_text_domain),
	__('Url', $rdetector_text_domain),
	__('Welcome message', $rdetector_text_domain),
	__('Icon', $rdetector_text_domain),
	__('Optional', $rdetector_text_domain),
	__('This entry is active', $rdetector_text_domain),
	__('Add this!', $rdetector_text_domain),
	$rdetector_plugin_dir
	);

	// the exluded urls
	printf('
		<div id="divExcluded">
			<p class="toggler"><span style="font-size:11px; font-weight:normal">%s</span></p>
			<div class="toggled help">%s</div>

			<form action="index.php" method="post" autocomplete="off" class="generic_form">
				<p><textarea name="rdetector_excluded_urls">%s</textarea></p>
				<div class="submit">
					<input type="hidden" name="rdetector_action" value="exclude" />
					<input type="submit" value="%s" />
				</div>
			</form>
		</div>',
		__('Get help with this', $rdetector_text_domain),
		__($rdetector_exclude_help, $rdetector_text_domain),
		get_option('rdetector_excluded_urls'),
		__('Exclude these out!', $rdetector_text_domain));

	// the generic options div
	printf('
	<div id="divGenericOptions">
		<p class="toggler"><span style="font-size:11px; font-weight:normal">%s</span></p>
		<div class="toggled help">%s</div>
		</h3>
		<form action="index.php" method="post" autocomplete="off" class="generic_form">',
		__('Get help with this', $rdetector_text_domain),
		__($rdetector_generic_help, $rdetector_text_domain));

	global $rdetector_options;

	foreach ($rdetector_options as $key => $config)
	{
		$value = get_option($key);
		if (!$value) $value = $config['default_value'];

		$select_options = '';
		foreach ($config['options'] as $o)
		{
			$select_options .= sprintf('
			<option value="%s"%s>%s</option>
			', $o, $o == $value ? 'selected="selected"' : '', $o);
		}
		printf('
		<p>
			<label for="%s">%s</label><br />
			<select name="%s" id="%s">
			%s
			</select>
		</p>', $key, $config['label'], $key, $key, $select_options);
	}

	printf('
			<div class="help">
				%s
				<input type="hidden" name="rdetector_action" value="generic" />
			</div>
			<div class="submit">
				<input type="submit" value="%s" />
			</div>
		</form>
	</div>',
	__("(*) If set to No, you may use a <a href=\"$rdetector_help_url#custom_tag\">custom tag</a> to manually display the welcome message on your posts.<br />
	Alternatively, you may put a <a href=\"http://www.phoenixheart.net/2008/11/here-it-is-referrer-detector-20/#template_tag\">template tag</a> in your template files (most likely index.php)."),
	__('Save these options', $rdetector_text_domain));

	// the stats div
	printf('
		<div id="divStats">
			<div id="statsHolder"></div>
			<div id="statsLoading" style="display: none"><img src="%simages/loading.gif" alt="Loading" /></div>
			<form id="formStats" method="post" action="index.php" autocomplete="off">
				<p>Time range:
					<select id="timeRange" name="time_range">
						<option value="today" selected="selected">%s</option>
						<option value="yesterday">%s</option>
						<option value="this_week">%s</option>
						<option value="this_month">%s</option>
						<option value="this_year">%s</option>
						<option value="all_time">%s</option>
						<option value="specified">%s</option>
					</select>
					<span id="specifiedSpan" style="display:none">
						%s <input type="text" title="YYYY-MM-DD" style="width:100px" name="specified_from" class="text" value="YYYY-DD-MM" />
						%s <input type="text" title="YYYY-MM-DD" style="width:100px" name="specified_to" class="text" value="YYYY-DD-MM" />
					</span>
					<input type="hidden" name="rdetector_action" value="stats_show" />
					<input type="submit" class="button-secondary action" value="%s" />
				</p>
			</form>
		</div>
	', $rdetector_plugin_dir,
	__('Today', $rdetector_text_domain),
	__('Yesterday', $rdetector_text_domain),
	__('This week', $rdetector_text_domain),
	__('This month', $rdetector_text_domain),
	__('This year', $rdetector_text_domain),
	__('All time (may take a while!)', $rdetector_text_domain),
	__('Specified', $rdetector_text_domain),
	__('From', $rdetector_text_domain),
	__('To', $rdetector_text_domain),
	__('Show Stats', $rdetector_text_domain));

	printf('
		<div id="divGenericLoading" style="display: none">
			<img src="%s/images/loading.gif" alt="Loading..." />
		</div>
	</div>', $rdetector_plugin_dir);
}

function rdetector_get_entry_table()
{
	global $rdetector_table_name, $wpdb, $rdetector_text_domain;

	$sql = "SELECT * FROM $rdetector_table_name ORDER BY `name`";
	$entries = $wpdb->get_results($sql);
	$table = sprintf('
	<table class="widefat" id="referrerDetectorEntries">
			<thead>
				<tr>
					<th class="check-column" scope="col"><input type="checkbox" autocomplete="off" /></th>
					<th>Name</th>
					<th>Url</th>
					<th>Icon</th>
					<th>Welcome Message</th>
					<th style="width: 200px; text-align: right">Action</th>
				</tr>
			</thead>
			<tbody>',
		__('Name', $rdetector_text_domain),
		__('Url', $rdetector_text_domain),
		__('Icon', $rdetector_text_domain),
		__('Welcome Message', $rdetector_text_domain),
		__('Action', $rdetector_text_domain));

	foreach ($entries as $entry)
	{
		$table .= rdetector_format_entry_html($entry->id, $entry->name, $entry->url, $entry->icon, $entry->welcome_text, $entry->enabled == 1);
	}

	$table .= '
			</tbody>
		</table>';

	return $table;
}

/**
 * @desc	Adds the Options menu item
 */
function rdetector_menu_items()
{
	global $rdetector_text_domain;

	add_options_page(
		__('Referrer Detector', $rdetector_text_domain),
		__('Referrer Detector', $rdetector_text_domain),
		8,
		basename(__FILE__),
		'rdetector_options_form'
	);
}

# Hook into the settings menu
add_action('admin_menu', 'rdetector_menu_items');

/**
 * @desc	Handles the form POST data to save the options
 */
function rdetector_request_handler()
{
	if (!isset($_POST['rdetector_action'])) return false;
	if ($_POST['rdetector_action'] == 'stats_insert') return false;

	global $rdetector_text_domain;

	if (!wp_verify_nonce($_POST['nonce'], RDETECTOR_NONCE_ACTION))
	{
		die(__('Security check failed. Please try refreshing.', $rdetector_text_domain));
	}

	switch($_POST['rdetector_action'])
	{
		case 'activate':
		case 'deactivate':
			rdetector_activate_entry($_POST['rdetector_action'] == 'activate');
			break;
		case 'bulk-activate':
		case 'bulk-deactivate':
			rdetector_activate_entry($_POST['rdetector_action'] == 'bulk-activate', true);
			break;
		case 'bulk-merge':
			rdetector_merge_entry();
			break;
		case 'add':
			rdetector_add_entry();
			break;
		case 'edit':
			rdetector_edit_entry();
			break;
		case 'update':
			rdetector_edit_entry(true);
			break;
		case 'delete':
			redetector_delete_entry();
			break;
		case 'bulk-delete':
			redetector_delete_entry(true);
			break;
		case 'restore':
			redetector_restore_default_entries();
			break;
		case 'exclude':
			// for now treat the excluded data as generic
		case 'generic':
			redetector_save_generic_options();
			break;
		case 'stats_show':
			rdetector_generate_stats_data();
			break;
		default:
			return false;
	}

	exit();
}

# Hook into the init action to handle the options request
add_action('init', 'rdetector_request_handler', 5);

function rdetector_public_request_handler()
{
	switch($_POST['rdetector_action'])
	{
		case 'stats_insert':
			rdetector_insert_stats();
			break;
		default:
			return false;
	}
	exit();
}

add_action('init', 'rdetector_public_request_handler', 5);

function rdetector_merge_entry()
{
	global $wpdb, $rdetector_table_name;

	$sql = sprintf("SELECT * FROM `$rdetector_table_name` WHERE `id` IN (%s)", mysql_escape_string($_POST['ids']));
	$result = $wpdb->get_results($sql);

	if (count($result) < 2)
	{
		rdetector_send_fake_error();
	}

	$to_delete = array();
	$new_url = array();
	foreach ($result as $entry)
	{
		$new_url[] = $entry->url;
		$to_delete[] = $entry->id;
	}

	$first_id = array_shift($to_delete);
	$wpdb->query(sprintf("DELETE FROM `$rdetector_table_name` WHERE `id` IN (%s)", implode(',', $to_delete)));
	$wpdb->query(sprintf("UPDATE `$rdetector_table_name` SET url='%s' WHERE `id`=$first_id", implode(',', $new_url)));

	rdetector_generate_js_data();
	echo rdetector_get_entry_table();
}

function rdetector_add_entry()
{
	// a little validation
	if (!$errors = rdetector_validate_post_data())
	{
		rdetector_send_fake_error();
	}

	global $wpdb, $rdetector_table_name;

	$message = rdetector_strip_tags($_POST['message']);
	$sql = $wpdb->prepare("INSERT INTO `$rdetector_table_name`(`name`, `url`, `icon`, `welcome_text`, `enabled`) VALUES(%s, %s, %s, %s, %d)",
			$_POST['name'], $_POST['url'], $_POST['icon'], $message, isset($_POST['enabled']));

	$wpdb->query($sql);

	rdetector_generate_js_data();
	echo rdetector_format_entry_html($wpdb->insert_id, $_POST['name'], $_POST['url'], $_POST['icon'], $message, isset($_POST['enabled']));
}

/**
* @desc 	Validates the POST data
*/
function rdetector_validate_post_data($is_editing = false)
{
	global $rdetector_text_domain;

	$errors = array();
	if (!isset($_POST['name']) || !trim($_POST['name']))
	{
		$errors[] = __('Entry name is required', $rdetector_text_domain);
	}
	if (!isset($_POST['url']) || !trim($_POST['url']))
	{
		$errors[] = __('URL is required', $rdetector_text_domain);
	}
	if (!isset($_POST['message']) || !trim($_POST['message']))
	{
		$errors[] = __('Welcome message is required', $rdetector_text_domain);
	}
	if ($is_editing && (!isset($_POST['id']) || !trim($_POST['id'])))
	{
		$errors[] = __('You must specify an ID', $rdetector_text_domain);
	}

	if (empty($errors)) return true;

	printf ('<p>%s</p>
	<ul>
		<li>%s</li>
	</ul>', __('Your input is not valid:', $rdetector_text_domain),
	implode('</li><li>', $errors));

	return false;
}

function rdetector_format_entry_html($id, $name, $url, $icon, $welcome_text, $enabled)
{
	global $rdetector_plugin_dir, $rdetector_text_domain;
	return sprintf('
		<tr class="%s" id="entry_%s">
			<td><input name="checked[]" class="entry-check" value="%s" type="checkbox" autocomplete="off" /></td>
			<td>%s</td>
			<td>%s</td>
			<td><img src="%s" alt="" /></td>
			<td>%s</td>
			<td style="text-align: right">
				<a href="#" entry="%s" class="ajax" action="%s" >%s</a> |
				<a href="#" entry="%s" class="ajax" action="edit">%s</a> |
				<a href="#" entry="%s" class="ajax need_confirm" confirmtext="%s" action="delete">%s</a>
				<span id="loading_%s" style="display:none"><img src="%s/images/loading.gif" alt="Loading..." /></span>
			</td>
		</tr>%s',
		$enabled ? __('active', $rdetector_text_domain) : __('inactive', $rdetector_text_domain), $id,
		$id,
		stripslashes($name),
		$url,
		$icon ? $icon : "{$rdetector_plugin_dir}images/icons/default.png",
		stripslashes($welcome_text),
		$id, $enabled ? 'deactivate' : 'activate', $enabled ? __('Deactivate', $rdetector_text_domain) : __('Activate', $rdetector_text_domain),
		$id, __('Edit/View', $rdetector_text_domain),
		$id, __('Are you sure you want to delete this entry?', $rdetector_text_domain), __('Delete', $rdetector_text_domain),
		$id, $rdetector_plugin_dir,
		PHP_EOL);
}

/**
* @desc 	(De)Activates an entry / entries (bulk mode)
*/
function rdetector_activate_entry($is_activating, $bulk = false)
{
	global $wpdb, $rdetector_table_name, $rdetector_text_domain;

	if ($bulk)
	{
		$sql = sprintf('UPDATE `%s` SET `enabled`=%d WHERE `id` IN (%s)', $rdetector_table_name, $is_activating ? 1 : 0, mysql_escape_string($_POST['ids']));
	}
	else
	{
		$sql = $wpdb->prepare('UPDATE `%s` SET `enabled`=%d WHERE `id`=%d', $rdetector_table_name, $is_activating ? 1 : 0, $_POST['id']);
	}
	$wpdb->query($sql);
	if (!mysql_affected_rows())
	{
		rdetector_send_fake_error();
	}

	rdetector_generate_js_data();

	if ($bulk)
	{
		die (rdetector_get_entry_table());
	}
	else
	{
		$is_activating ? _e('Deactivate', $rdetector_text_domain) : _e('Activate', $rdetector_text_domain);
	}
}

/**
* @desc 	Edits an entry
*/
function rdetector_edit_entry($is_updating = false)
{
	global $wpdb, $rdetector_table_name, $rdetector_text_domain;

	$id = intval($_POST['id']);
	if (!$is_updating)
	{
		$sql = "SELECT * FROM `$rdetector_table_name` WHERE `id`=$id";
		$entry = $wpdb->get_row($sql);
		if (empty($entry))
		{
			rdetector_send_fake_error();
		}
		printf('<p>
			<label for="name">%s</label><br />
			<input type="text" name="name" id="name" class="required" value="%s" />
		</p>
		<p>
			<label for="url">%s</label><br />
			<input type="text" name="url" id="url" class="required" value="%s" />
		</p>
		<p>
			<label for="message">%s</label><br />
			<textarea name="message" id="message" class="required">%s</textarea>
		</p>
		<p>
			<label for="icon">%s</label><br />
			<input type="text" name="icon" id="icon" value="%s" /> (%s)
		</p>
		<p>
			<input type="checkbox" name="enabled" id="enabled" %s />
			<label for="enabled">%s</label>
			<input type="hidden" name="rdetector_action" value="update" />
			<input type="hidden" name="id" value="%s" />
		</p>
		<div class="submit">
			<input type="submit" value="%s" />
			<input type="button" id="buttonCancel" value="%s" />
		</div>',
		__('Name', $rdetector_text_domain),
		addslashes($entry->name),
		__('Url', $rdetector_text_domain),
		$entry->url,
		__('Welcome message', $rdetector_text_domain),
		stripslashes(htmlentities($entry->welcome_text)),
		__('Icon', $rdetector_text_domain),
		$entry->icon, __('Optional', $rdetector_text_domain),
		$entry->enabled == 1 ? 'checked="checked"' : '',
		__('This entry is active', $rdetector_text_domain),
		$entry->id,
		__('Save it!', $rdetector_text_domain),
		__('Cancel', $rdetector_text_domain));
	}
	else
	{
		if (!$errors = rdetector_validate_post_data(true))
		{
			return false;
		}

		global $wpdb, $rdetector_table_name;

		$message = rdetector_strip_tags($_POST['message']);
		$sql = $wpdb->prepare("UPDATE `$rdetector_table_name` SET `name`=%s, `url`=%s, `icon`=%s, `welcome_text`=%s, `enabled`=%d WHERE `id`=%d",
			$_POST['name'], $_POST['url'], $_POST['icon'], $message, isset($_POST['enabled']), intval($_POST['id']));

		$wpdb->query($sql);

		rdetector_generate_js_data();

		echo rdetector_format_entry_html(intval($_POST['id']), $_POST['name'], $_POST['url'], $_POST['icon'], $message, isset($_POST['enabled']));
	}
}

function redetector_delete_entry($bulk = false)
{
	global $wpdb, $rdetector_table_name;

	if ($bulk)
	{
		$sql = sprintf("DELETE FROM `$rdetector_table_name` WHERE `id` IN (%s)", mysql_escape_string($_POST['ids']));
	}
	else
	{
		$sql = $wpdb->prepare("DELETE FROM `$rdetector_table_name` WHERE `id`=%d LIMIT 1", $_POST['id']);
	}

	$wpdb->query($sql);

	if (!mysql_affected_rows())
	{
		rdetector_send_fake_error();
	}

	rdetector_generate_js_data();
}

/**
* @desc 	Restore the default entries
*/
function redetector_restore_default_entries()
{
	global $wpdb, $rdetector_table_name;

	$wpdb->query("TRUNCATE $rdetector_table_name");
	rdetector_add_default_entries();
	rdetector_generate_js_data();
	echo rdetector_get_entry_table();
}

/**
* @desc 	Saves the generic options of the plugin
*/
function redetector_save_generic_options()
{
	// we have 2 cases
	if (isset($_POST['rdetector_excluded_urls']))
	{
		// 1. Excluded URLS
		$urls = explode(',', $_POST['rdetector_excluded_urls']);
		array_walk($urls, 'rdetector_trim_value');
		update_option('rdetector_excluded_urls', implode(',', $urls));
	}
	else
	{
		// 2. "Real" generic options
		global $rdetector_options, $rdetector_text_domain;

		foreach ($rdetector_options as $key => $config)
		{
			if (isset($_POST[$key]))
			{
				update_option($key, $_POST[$key]);
			}
		}
	}

	rdetector_generate_js_data();
	_e("Settings saved.", $rdetector_text_domain);
}

function rdetector_trim_value(&$value)
{
    $value = trim($value);
}

/**
* @desc 	(Re)Generates the data in JS format
*/
function rdetector_generate_js_data()
{
	// if ($_SERVER['REMOTE_ADDR'] == '127.0.0.1') return true;

	global $wpdb, $rdetector_table_name, $rdetector_options, $rdetector_plugin_dir, $rdetector_text_domain;
	$sql = "SELECT * FROM `$rdetector_table_name` WHERE `enabled`=1";
	$entries = $wpdb->get_results($sql);

	// the entries
	$data[] = 'var rdetector_entries = new Array();';
	foreach ($entries as $entry)
	{
		if (empty($entry->icon)) $entry->icon = "{$rdetector_plugin_dir}images/icons/default.png";
		$key = base64_encode($entry->url);
		$data[] = sprintf('rdetector_entries["%s"] = new Array();', $key);
		$data[] = sprintf('rdetector_entries["%s"]["message"] = "%s";', $key, base64_encode(stripslashes($entry->welcome_text)));
		$data[] = sprintf('rdetector_entries["%s"]["icon"] = "%s";', $key, base64_encode($entry->icon));
	}

	// the options
	$data[] = 'var rdetector_options = new Array();';
	foreach ($rdetector_options as $key => $config)
	{
		$value = get_option($key);
		$data[] = sprintf('rdetector_options["%s"] = "%s";', $key, empty($value) ? $config['default_value'] : $value);
	}
	// don't forget the excluded URLs
	$data[] = sprintf('rdetector_options["rdetector_excluded_urls"] = "%s";', base64_encode(get_option('rdetector_excluded_urls')));

	// save
	if (!$handle = fopen("../wp-content/plugins/referrer-detector/js/data.js", 'w'))
	{
		exit(__("Error: Could not open wp-content/plugins/referrer-detector/js/data.js for writing, please check file permission.", $rdetector_text_domain));
	}

	fwrite($handle, implode(PHP_EOL, $data));
	fclose($handle);
}

/**
 * @desc	Inserts a stat entry
 */
function rdetector_insert_stats()
{
	global $wpdb, $rdetector_stats_table_name;
	$sql = sprintf("INSERT INTO $rdetector_stats_table_name(`referrer`, `time`) VALUES('%s', now())",
		base64_decode($_POST['r']));
	$wpdb->query($sql);
}

/**
 * @desc	Generates the stat data for a specific time
 */
function rdetector_generate_stats_data()
{
	global $rdetector_stats_table_name, $wpdb, $rdetector_text_domain;

	switch ($_POST['time_range'])
	{
		case 'today':
			$start_date = date('Y-m-d 00:00:00');
			$end_date =  date('Y-m-d H:i:s', strtotime('tomorrow'));
			break;
		case 'yesterday':
			$start_date = date('Y-m-d H:i:s', strtotime('yesterday'));
			$end_date = date('Y-m-d 00:00:00'); // today
			break;
		case 'this_week':
			$start_date = date('Y-m-d H:i:s', strtotime('today') - (date('w') - 1)*60*60*24);
			$end_date =  date('Y-m-d H:i:s', strtotime('tomorrow'));
			break;
		case 'this_month':
			$start_date = date('Y-m-01 00:00:00');
			$end_date =  date('Y-m-d H:i:s', strtotime('tomorrow'));
			break;
		case 'this_year':
			$start_date = date('Y-m-01 00:00:00');
			$end_date =  date('Y-m-d H:i:s', strtotime('tomorrow'));
			break;
		case 'all_time':
			$start_date = '2008-01-01 00:00:00';
			$end_date =  date('Y-m-d H:i:s', strtotime('tomorrow'));
			break;
		case 'specified':
			preg_match('#([0-9]{1,4})-([0-9]{1,2})-([0-9]{1,2})#', $_POST['specified_to'], $matches);
			if (empty($matches))
			{
				$the_date_after = strtotime('tomorrow');
			}
  			else
  			{
  				$the_date_after = mktime(0, 0, 0, $matches[2], $matches[3], $matches[1]) + 24*60*60;
  			}

			$start_date = mysql_escape_string($_POST['specified_from']);
			$end_date = date('Y-m-d 00:00:00', $the_date_after);
			break;
		default:
			break;
	}

	$sql = "SELECT `referrer`, COUNT(`id`) AS 'views'
		FROM `$rdetector_stats_table_name`
		WHERE `time` >= '$start_date'
		AND `time` < '$end_date'
		GROUP BY `referrer`
		ORDER BY `referrer`";

	// echo "$sql <br />";

	$entries = $wpdb->get_results($sql);
	if (count($entries) == 0)
	{
		die(__('No stats data for the time range, or the range is not so valid.', $rdetector_text_domain));
	}

	// write the data into the xml file
	$data = '';
	foreach ($entries as $entry)
	{
		$data .= "<row>
	<string>$entry->referrer</string>
	<number tooltip='$entry->referrer'>$entry->views</number>
</row>
";
	}

	$data = "<chart>
	<chart_data>
		<row>
			<string></string>
			<string></string>
		</row>
	$data
	</chart_data>
	<chart_type>bar</chart_type>
	<chart_label position='right'
                font='arial'
                bold='false'
                color='FFFFFF'
				background_color='000000'
                alpha='90'
				size='11'/>
	<legend size='10'
			layout='vertical'
			x='10'
			y='10'
			width='150'
			bold='false'
			transition='slide_down' />
	<chart_rect x='170' y='10' width='500' />
	<chart_transition type='scale' order='series' duration='0.5' />
	<series set_gap='10' />
</chart>";

	if (!$handle = fopen("../wp-content/plugins/referrer-detector/charts/data.xml", 'w'))
	{
		exit(__("Error: Could not open wp-content/plugins/referrer-detector/charts/data.xml for writing, please check file permission.", $rdetector_text_domain));
	}
	fwrite($handle, $data);
	fclose($handle);

	exit('<embed width="700" height="400"
	scale="noscale"
	salign="TL"
	bgcolor="#cccccc"
	wmode="opaque"
	src="../wp-content/plugins/referrer-detector/charts/charts.swf"
	FlashVars="library_path=../wp-content/plugins/referrer-detector/charts/charts_library&xml_source=../wp-content/plugins/referrer-detector/charts/data.xml?c="' . microtime() . '
	name="my_chart"
	menu="true"
	allowFullScreen="true"
	allowScriptAccess="sameDomain"
	quality="high"
	align="middle"
	pluginspage="http://www.macromedia.com/go/getflashplayer"
	play="true"
	devicefont="false"
	type="application/x-shockwave-flash" >
	</embed>');
}

/**
* @desc 	Strips unallowed tags from the welcome message
*/
function rdetector_strip_tags($message)
{
	$allowable_tags = '<div><h1><h2><h3><p><span><a><ul><ol><li><hr><br><table><thead><tbody><tfoot><tr><td><th><strong><em>';
	return $message = strip_tags(html_entity_decode($message), $allowable_tags);
}

/**
* @desc 	Sends a fake error (for AJAX purpose)
*/
function rdetector_send_fake_error()
{
	header('HTTP/1.0 404 Not Found');
	exit();
}

function rdetector_prepare_post_data($post_content)
{
	// don't display the greeting if it's a feed
	if (is_feed() || is_home())
	{
		return str_replace(RDETECTOR_CUSTOM_TAG, '', $post_content);
	}

	global $post;
	$url = get_permalink($post->ID);
	$title = get_the_title();
	$author_name = get_the_author();
	$author_url = get_the_author_url();
	$category_names = array();
	$category_links = array();

	foreach((get_the_category()) as $category)
	{
		$category_names[] = $category->cat_name;
		$category_links[] = sprintf('<a href="%s">%s</a>', get_category_link($category->cat_ID), $category->cat_name);
	}

	$tags = array(
		'url' 			=> $url,
		'title'			=> $title,
		'link'			=> sprintf('<a href="%s">%s</a>', $url, $title),
		'categorynames'	=> implode(', ', $category_names),
		'categorylinks'	=> implode(', ', $category_links),
		'authorname' 	=> $author_name,
		'authorurl'		=> $author_url,
		'authorlink'	=> sprintf('<a href="%s">%s</a>', $author_url, $author_name),
	);

	// merge all the data into a hidden input's value
	$data = '';
	foreach ($tags as $key => $value)
	{
		$data .= "$key:" . base64_encode($value) . ',';
	}
	$data = sprintf('<input type="hidden" id="rdetector_data" value="%s" name="rdetector_data" />', rtrim($data, ','));
	$data .= sprintf('<input type="hidden" id="rdetector_post_page" value="%s" name="rdetector_post_page" />', is_page() ? 'page' : 'post');

	$post_content .= $data;

	$post_content = str_replace(RDETECTOR_CUSTOM_TAG, '<span class="rdetector_placeholder"></span>', $post_content);

	return '<span class="rdetector_placeholder_before"></span>' . $post_content . '<span class="rdetector_placeholder_after"></span>';
}

add_filter('the_content', 'rdetector_prepare_post_data', 5);

/**
* @desc 	Includes the stylesheet for the greet box
*/
function rdetector_head()
{
	global $rdetector_plugin_dir;

	printf('<link rel="stylesheet" type="text/css" media="screen" href="%scss/style.css" />', $rdetector_plugin_dir);
	print('<script type="text/javascript" src="http://jqueryjs.googlecode.com/files/jquery-1.2.6.min.js"></script>');
	printf('<script type="text/javascript" src="%sjs/data.js"></script>', $rdetector_plugin_dir);
	printf('<script type="text/javascript" src="%sjs/utils.js"></script>', $rdetector_plugin_dir);
	printf('<script type="text/javascript" src="%sjs/onload.js"></script>', $rdetector_plugin_dir);
}

add_action('wp_head', 'rdetector_head');

/**
* @desc
*/
function referrer_detector()
{
	echo '<div class="rdetector_placeholder_special" style="display: none"></div>';
}